// /////////////////////////////////////////////////////////////////////////////
// DR DOBB'S CHALLENGES
//
// Filename       : game.cpp
// Date           : February 2008
//
// Description    : Refer to description in corresponding header.
//
// ///////////////////////////////////////////////////////////////////////////


#include <tchar.h>

#include "game.h"
#include "input.h"

#include "EntityTokenCollector.h"
#include "EntityThrownToken.h"
#include "EntityLaser.h"
#include "EntityShodan.h"
#include "EntitySwitch.h"
#include "EntityEye.h"

#include "Application.h"

#include "Util.h"
#include "Extro.h"



Game::Game() :
  m_GetReadyDelay( 0.0f ),
  m_Level( NULL ),
  m_AlertActive( false ),
  m_AlertPos( 0.0f ),
  m_SubShotDir( Dobbs::DIR_RIGHT ),
  m_GameWon( false ),
  m_WinDelay( 0.0f )
{

  m_pSoundMessage = NULL;
  m_CollectedIcons    = 0;
  m_PlayerStartTokens = 0;

  m_pEntityTokens = Entity::SpawnEntity( Dobbs::ENTITY_TYPE_TOKEN, 22, 22, Dobbs::DIR_NONE );

  m_Level.m_pGame = this;

  m_LevelNr     = 1;

  for ( int i = 0; i < 4; ++i )
  {
    m_PreItemCollected[i] = false;
    m_ItemCollected[i] = false;
  }

}



Game::~Game() 
{

  delete m_pEntityTokens;
  delete m_pSoundMessage;

}



void Game::Init()
{

  m_pPlayer     = NULL;

  // Load level entities
  LoadLevel();

  m_pPlayer = (EntityPlayer*)m_Level.SpawnEntity( Dobbs::ENTITY_TYPE_PLAYER, 0, 164, 350 );
  PanOnPlayer();

  m_PlayerStartX = m_pPlayer->GetX();
  m_PlayerStartY = m_pPlayer->GetY();

  m_Paused    = false;

  m_pPlayer->WarpIn();

}



void Game::Exit()
{

  g_App.StopSound( "Sub.Motor" );

}



void Game::LoadLevel() 
{

  m_Level.Load( m_LevelNr );

  std::list<Entity*>::iterator    it( m_Level.m_Entities.begin() );
  while ( it != m_Level.m_Entities.end() )
  {
    Entity*   pEntity( *it );

    // remove already collected items
    if ( ( m_PreCollectedItems[m_LevelNr].find( std::make_pair( pEntity->GetX() / 32, pEntity->GetY() / 32 ) ) != m_PreCollectedItems[m_LevelNr].end() )
    ||   ( m_CollectedItems[m_LevelNr].find( std::make_pair( pEntity->GetX() / 32, pEntity->GetY() / 32 ) ) != m_CollectedItems[m_LevelNr].end() ) )
    {
      if ( pEntity->GetType() == Dobbs::ENTITY_TYPE_TOKEN_COLLECTOR )
      {
        EntityTokenCollector*   pCollector = (EntityTokenCollector*)pEntity;
        ActivateCollector( pCollector );
        pCollector->m_TokensNeeded = 0;

        ++it;
      }
      else if ( pEntity->GetType() == Dobbs::ENTITY_TYPE_EYE )
      {
        // make sure the alarm doesn't get off again
        EntityEye*   pEye = (EntityEye*)pEntity;

        pEye->Wakeup();

        ++it;
      }
      else
      {
        delete pEntity;
        it = m_Level.m_Entities.erase( it );
      }
    }
    else if ( pEntity->GetType() == Dobbs::ENTITY_TYPE_SUB )
    {
      // restore the subs position when returning
      std::map<int,std::pair<int,int> >::iterator   itS( m_StoredSubPositions.find( m_LevelNr ) );
      if ( itS != m_StoredSubPositions.end() )
      {
        pEntity->SetPosition( itS->second.first, itS->second.second );
      }
      ++it;
    }
    else
    {
      ++it;
    }
  }

  // Security Pass
  for ( int i = 0; i < 4; ++i )
  {
    if ( ( m_PreItemCollected[i] )
    ||   ( m_ItemCollected[i] ) )
    {
      Dobbs::TileType   OldDoor = Dobbs::TILE_DOOR_YELLOW;
      switch ( i )
      {
        case 1:
          OldDoor = Dobbs::TILE_DOOR_RED;
          break;
        case 2:
          OldDoor = Dobbs::TILE_DOOR_BLUE;
          break;
        case 3:
          OldDoor = Dobbs::TILE_DOOR_GREEN;
          break;
      }
      m_Level.ReplaceTile( OldDoor, Dobbs::TILE_DOOR_OPENED );
    }
  }
  if ( !m_Level.m_Name.empty() )
  {
    m_GetReadyDelay = 2.5f;
  }

  if ( m_pPlayer )
  {
    m_pPlayer->Unhide();
  }

  m_Level.EnableLayer( 0 );
  m_Level.EnableLayer( 1 );
  m_Level.EnableLayer( 2 );

  g_App.StopSound( "Sub.Motor" );

}



void Game::UpdateTimed( const float ElapsedTime )
{

  if ( m_WinDelay > 0.0f )
  {
    m_WinDelay -= ElapsedTime;
    if ( m_WinDelay <= 0.0f )
    {
      g_App.ChangeGameState( new Extro() );
      return;
    }
  }

  if ( m_AlertActive )
  {
    m_AlertPos += 100.0f * ElapsedTime;
  }

  if ( m_GetReadyDelay > 0.0f )
  {
    m_GetReadyDelay -= ElapsedTime;
    if ( m_GetReadyDelay <= 0.0f )
    {
      m_GetReadyDelay = 0.0f;
    }
  }
  else if ( !m_Paused )
  {
    /*
    if ( ( m_pPlayer )
    &&   ( g_App.m_Input.KeyDown( 'Q' ) ) )
    {
      if ( g_App.m_Input.KeyDown( VK_LEFT ) )
      {
        m_pPlayer->SetPosition( m_pPlayer->GetX() - 10, m_pPlayer->GetY() );
      }
      if ( g_App.m_Input.KeyDown( VK_RIGHT ) )
      {
        m_pPlayer->SetPosition( m_pPlayer->GetX() + 10, m_pPlayer->GetY() );
      }
      if ( g_App.m_Input.KeyDown( VK_UP ) )
      {
        m_pPlayer->SetPosition( m_pPlayer->GetX(), m_pPlayer->GetY() - 10 );
      }
      if ( g_App.m_Input.KeyDown( VK_DOWN ) )
      {
        m_pPlayer->SetPosition( m_pPlayer->GetX(), m_pPlayer->GetY() + 10 );
      }
    }
    */

    if ( ( m_pPlayer )
    &&   ( !m_GameWon ) )
    {
      /*
      if ( g_App.m_Input.KeyPressed( 'C' ) )
      {
        g_App.ChangeGameState( new Extro() );
        //OnGameEvent( Dobbs::GE_SHODAN_KILLED );


        //Warp( 12, 218, 1435 );
        //Warp( 11, 903, 254 );
        //Warp( 10, 272, 989 );
        //Warp( 9, 1305, 2462 );
        return;
      }
      if ( g_App.m_Input.KeyPressed( '1' ) )
      {
        m_PreItemCollected[0] = true;
        m_ItemCollected[0] = true;

        m_Level.ReplaceTile( Dobbs::TILE_DOOR_YELLOW, Dobbs::TILE_DOOR_OPENED );
        return;
      }
      */

      if ( g_App.m_Input.KeyPressed( 'R' ) )
      {
        Restart();
        return;
      }

      if ( m_pPlayer->GetState() == Dobbs::STATE_OK ) 
      {
        // Check if player is still alive...

        if ( g_App.m_Input.KeyDown( VK_RIGHT ) )
        {
          m_pPlayer->SetDirection( Dobbs::DIR_RIGHT );
          if ( m_pPlayer->Move( m_Level, 210.0f * ElapsedTime, 0.0f, Dobbs::DIR_RIGHT ) == Dobbs::BLOCKED_NONE )
          {
            if ( m_pPlayer->OnGround() )
            {
              m_pPlayer->SetFrame( Dobbs::ANIM_PLAYER_RUN_RIGHT );
              m_pPlayer->RunSound();
            }
          }
          else
          {
            m_pPlayer->RunSound( false );
          }
        }
        else if ( g_App.m_Input.KeyDown( VK_LEFT ) )
        {
          m_pPlayer->SetDirection( Dobbs::DIR_LEFT );
          if ( m_pPlayer->Move( m_Level, -210.0f * ElapsedTime, 0.0f, Dobbs::DIR_LEFT ) == Dobbs::BLOCKED_NONE )
          {
            if ( m_pPlayer->OnGround() )
            {
              m_pPlayer->SetFrame( Dobbs::ANIM_PLAYER_RUN_LEFT );
              m_pPlayer->RunSound();
            }
          }
          else
          {
            m_pPlayer->RunSound( false );
          }
        }

        if ( g_App.m_Input.KeyPressed( VK_UP ) )
        {
          if ( m_pPlayer->Jump( m_Level, 660 ) )
          {
          }
        }
        if ( g_App.m_Input.KeyDown( VK_DOWN ) )
        {
          m_pPlayer->Drop( m_Level );
        }
      }
      else if ( m_pPlayer->GetState() == Dobbs::STATE_IN_VEHICLE ) 
      {
        if ( g_App.m_Input.KeyPressed( VK_UP ) )
        {
          if ( m_pPlayer->Carrier()->Jump( m_Level, 660 ) )
          {
            g_App.PlaySound( "Player.Jump" );
          }
        }
        else if ( g_App.m_Input.KeyPressed( VK_DOWN ) )
        {
          // jump out of the vehicle
          if ( m_pPlayer->ForceJump( m_Level, 660 ) )
          {
            g_App.PlaySound( "Player.Jump" );
          }
        }
      }
      else if ( m_pPlayer->GetState() == Dobbs::STATE_IN_SUB ) 
      {
        bool    SubMoved = false;
        if ( g_App.m_Input.KeyDown( VK_UP ) )
        {
          float   VY = m_pPlayer->Carrier()->GetVelY();
          VY -= 5.0f;
          if ( VY < -120.0f )
          {
            VY = -120.0f;
          }
          m_pPlayer->Carrier()->SetVelY( VY );
          SubMoved = true;
        }
        if ( g_App.m_Input.KeyDown( VK_DOWN ) )
        {
          float   VY = m_pPlayer->Carrier()->GetVelY();
          VY += 5.0f;
          if ( VY > 120.0f )
          {
            VY = 120.0f;
          }
          m_pPlayer->Carrier()->SetVelY( VY );
          SubMoved = true;
        }
        if ( g_App.m_Input.KeyDown( VK_LEFT ) )
        {
          m_pPlayer->Carrier()->SetFrame( Dobbs::ANIM_SUB_L );
          float   VX = m_pPlayer->Carrier()->GetVelX();
          VX -= 5.0f;
          if ( VX < -120.0f )
          {
            VX = -120.0f;
          }
          m_pPlayer->Carrier()->SetVelX( VX );
          m_SubShotDir = Dobbs::DIR_LEFT;
          SubMoved = true;
        }
        if ( g_App.m_Input.KeyDown( VK_RIGHT ) )
        {
          m_pPlayer->Carrier()->SetFrame( Dobbs::ANIM_SUB_R );
          float   VX = m_pPlayer->Carrier()->GetVelX();
          VX += 5.0f;
          if ( VX > 120.0f )
          {
            VX = 120.0f;
          }
          m_pPlayer->Carrier()->SetVelX( VX );
          m_SubShotDir = Dobbs::DIR_RIGHT;
          SubMoved = true;
        }
        if ( SubMoved != g_App.IsSoundPlaying( "Sub.Motor" ) )
        {
          if ( SubMoved )
          {
            g_App.LoopSound( "Sub.Motor" );
          }
          else
          {
            g_App.StopSound( "Sub.Motor" );
          }
        }
      }
    }
    else if ( ( m_pPlayer )
    &&        ( m_GameWon ) )
    {
      if ( m_WinDelay <= 5.0f )
      {
        if ( m_pPlayer->GetState() != Dobbs::STATE_WARPING )
        {
          m_pPlayer->WarpOut();
        }
      }
    }
    m_Level.UpdateTimed( ElapsedTime );
    PanOnPlayer();
  }

}



void Game::ActivateCollector( EntityTokenCollector* pCollector )
{

  // find a door to open nearby
  int   TX = pCollector->GetX() / 32;
  int   TY = pCollector->GetY() / 32;

  for ( int i = -4; i <= 4; ++i )
  {
    if ( m_Level.Tile( TX + i, TY ) == Dobbs::TILE_METAL_DOOR )
    {
      // door found
      int   Y1 = TY;
      int   Y2 = TY;

      // find upper and lower extent of door
      while ( ( m_Level.Tile( TX + i, Y1 ) == Dobbs::TILE_METAL_DOOR )
      ||      ( m_Level.Tile( TX + i, Y1 - 1 ) == Dobbs::TILE_METAL_DOOR ) )
      {
        --Y1;
      }
      ++Y1;
      while ( ( m_Level.Tile( TX + i, Y2 ) == Dobbs::TILE_METAL_DOOR )
      ||      ( m_Level.Tile( TX + i, Y2 + 1 ) == Dobbs::TILE_METAL_DOOR ) )
      {
        ++Y2;
      }
      --Y2;
      for ( int j = Y1; j <= Y2; ++j )
      {
        if ( m_Level.Tile( TX + i, j ) == Dobbs::TILE_METAL_DOOR )
        {
          m_Level.SetTile( 1, TX + i, j, Dobbs::TILE_METAL_DOOR_OPEN );
        }
      }
    }
  }

}



void Game::Update() 
{
  if ( m_GetReadyDelay > 0.0f )
  {
    if ( g_App.m_Input.KeyPressed( VK_SPACE ) ) 
    {
      m_GetReadyDelay = 0.0000001f;
    }
    return;
  }

  if ( !m_Paused ) 
  {
    // Update all entities
    m_Level.Update();

    // If all tokens have been collected, end level
    // TODO
    /*
    if ( m_Level.IsDone() )
    {
      if ( !m_isVictory )
      {
        m_isVictory = true;
        m_pVictory->Unhide();
        m_pDimmer->Unhide();

        // Assert pauseGame to determine what type of inputs are active
        m_pauseGame = true;

        // Set pauseTime to the current ticks so startTime can be adjusted
        // when the game is unpaused such that time used during the pause
        // won't be included in the overall time elapsed
        m_pauseTime = GetTickCount();

        // Pause all sprites to ensure nothing is animating during the pause
        Sprite::PauseAll();

        m_Level.UpdateScore( m_pauseTime - m_startTime );
      }
    } 
    else */
    if ( g_App.m_Input.KeyPressed( VK_ESCAPE ) ) 
    {
      // Check for ESC to pause the game (and return to the menu)
      m_Paused = true;
    } 
    /*
    else if ( g_App.m_Input.KeyPressed( 'N' ) ) 
    {
      // next level
      m_LevelNr++;
      m_Level.m_Entities.remove( m_pPlayer );
      LoadLevel();
      m_Level.InsertEntity( m_pPlayer );
      m_Level.m_pPlayer = m_pPlayer;
      m_Level.m_pPlayer->WarpIn();
    }
    */
    else if ( ( m_pPlayer )
    &&        ( m_pPlayer->GetState() == Dobbs::STATE_OK ) )
    {
      // Check if player is still alive...

      // Check for LEFT/RIGHT running keys (right-run has precidence)
      if ( g_App.m_Input.KeyPressed( VK_SPACE ) )
      {
        bool    ActionDone = false;

        Entity*   pDoor = m_Level.FindCollidingEntity( m_pPlayer, Dobbs::ENTITY_TYPE_DOOR );
        if ( ( m_pPlayer->OnGround() )
        &&   ( pDoor ) )
        {
          WalkThroughDoor( pDoor );
          ActionDone = true;
        }
        EntityTokenCollector*   pCollector = (EntityTokenCollector*)m_Level.FindCollidingEntity( m_pPlayer, Dobbs::ENTITY_TYPE_TOKEN_COLLECTOR );
        if ( ( m_pPlayer->OnGround() )
        &&   ( pCollector )
        &&   ( pCollector->m_TokensNeeded != 0 )
        &&   ( !ActionDone ) )
        {
          if ( m_CollectedIcons >= pCollector->m_TokensNeeded )
          {
            m_CollectedIcons -= pCollector->m_TokensNeeded;
            pCollector->m_TokensNeeded = 0;

            m_PreCollectedItems[m_LevelNr].insert( std::make_pair( pCollector->GetSpawnX() / 32, pCollector->GetSpawnY() / 32 ) );

            ActivateCollector( pCollector );

            g_App.PlaySound( "Door.Open" );
            ActionDone = true;
          }
        }
        EntitySwitch*   pSwitch = (EntitySwitch*)m_Level.FindCollidingEntity( m_pPlayer, Dobbs::ENTITY_TYPE_SWITCH );
        if ( pSwitch )
        {
          if ( pSwitch->Switch() )
          {
            OnGameEvent( (Dobbs::GameEvent)pSwitch->m_ExtraData );
          }
          ActionDone = true;
        }
        if ( !ActionDone )
        {
          if ( m_CollectedIcons > 0 )
          {
            --m_CollectedIcons;
            Entity* pToken = NULL;
            if ( m_pPlayer->Direction() == Dobbs::DIR_LEFT )
            {
              pToken = m_Level.SpawnEntity( Dobbs::ENTITY_TYPE_THROWN_TOKEN, Dobbs::DIR_LEFT, m_pPlayer->GetX() - 20, m_pPlayer->GetY() - 70 );
              // make sure the token does not start inside a wall
              while ( m_Level.IsAreaBlocked( pToken, pToken->CollisionRect(), Dobbs::DIR_NONE ) )
              {
                pToken->SetPosition( pToken->GetX() + 1, pToken->GetY() + 1 );
              }
            }
            else
            {
              pToken = m_Level.SpawnEntity( Dobbs::ENTITY_TYPE_THROWN_TOKEN, Dobbs::DIR_RIGHT, m_pPlayer->GetX() + 20, m_pPlayer->GetY() - 70 );
              // make sure the token does not start inside a wall
              while ( m_Level.IsAreaBlocked( pToken, pToken->CollisionRect(), Dobbs::DIR_NONE ) )
              {
                pToken->SetPosition( pToken->GetX() - 1, pToken->GetY() + 1 );
              }
            }
            pToken->ForceJump( m_Level, 400 );
            g_App.PlaySound( "Player.Throw" );
          }
        }
      }
    }
    else if ( ( m_pPlayer )
    &&        ( m_pPlayer->GetState() == Dobbs::STATE_IN_VEHICLE ) )
    {
      if ( g_App.m_Input.KeyPressed( VK_SPACE ) )
      {
        bool    ActionDone = false;

        if ( m_CollectedIcons > 0 )
        {
          --m_CollectedIcons;
          Entity* pToken = NULL;
          if ( m_pPlayer->Direction() == Dobbs::DIR_LEFT )
          {
            pToken = m_Level.SpawnEntity( Dobbs::ENTITY_TYPE_THROWN_TOKEN, Dobbs::DIR_LEFT, m_pPlayer->GetX() - 20, m_pPlayer->GetY() - 70 );
            // make sure the token does not start inside a wall
            while ( m_Level.IsAreaBlocked( pToken, pToken->CollisionRect(), Dobbs::DIR_NONE ) )
            {
              pToken->SetPosition( pToken->GetX() + 1, pToken->GetY() + 1 );
            }
          }
          else
          {
            pToken = m_Level.SpawnEntity( Dobbs::ENTITY_TYPE_THROWN_TOKEN, Dobbs::DIR_RIGHT, m_pPlayer->GetX() + 20, m_pPlayer->GetY() - 70 );
            // make sure the token does not start inside a wall
            while ( m_Level.IsAreaBlocked( pToken, pToken->CollisionRect(), Dobbs::DIR_NONE ) )
            {
              pToken->SetPosition( pToken->GetX() - 1, pToken->GetY() + 1 );
            }
          }
          pToken->ForceJump( m_Level, 400 );
          g_App.PlaySound( "Player.Throw" );
        }
      }
    }
    else if ( ( m_pPlayer )
    &&        ( m_pPlayer->GetState() == Dobbs::STATE_IN_CANNON ) )
    {
      if ( g_App.m_Input.KeyPressed( VK_SPACE ) ) 
      {
        m_pPlayer->ShootFromCannon();
      }
    }
    else if ( ( m_pPlayer )
    &&        ( m_pPlayer->GetState() == Dobbs::STATE_IN_SUB ) )
    {
      if ( g_App.m_Input.KeyPressed( VK_SPACE ) )
      {
        bool    ActionDone = false;

        Entity* pShot = NULL;
        if ( m_SubShotDir == Dobbs::DIR_LEFT )
        {
          pShot = m_Level.SpawnEntity( Dobbs::ENTITY_TYPE_SUB_SHOT, Dobbs::DIR_LEFT, m_pPlayer->GetX() - 60, m_pPlayer->GetY() + 12 );
        }
        else
        {
          pShot = m_Level.SpawnEntity( Dobbs::ENTITY_TYPE_SUB_SHOT, Dobbs::DIR_RIGHT, m_pPlayer->GetX() + 22, m_pPlayer->GetY() + 12 );
        }
        g_App.PlaySound( "Sub.Laser" );
        if ( m_Level.IsAreaBlocked( pShot, pShot->CollisionRect(), Dobbs::DIR_NONE ) )
        {
          pShot->Die();
        }
      }
    }
    else if ( m_pPlayer == NULL )
    {
      // Check for SPACEBAR if game paused after DEATH (to restart)
      if ( g_App.m_Input.KeyPressed( VK_SPACE ) ) 
      {
        Restart();
      }
    }
  }
  else 
  {
    // Handle inputs when game is paused

    // Otherwise, check for Y/N if game paused after ESCAPE
    // If Y to return to main menu, adjust startTime first, then stop
    if ( g_App.m_Input.KeyPressed( 'Y' ) ) 
    {
      g_App.ChangeGameState( new Menu() );
      return;
    } 
    else if ( ( g_App.m_Input.KeyPressed( 'N' ) ) 
    ||        ( g_App.m_Input.KeyPressed( VK_ESCAPE ) ) )
    {
      // If N to return to game, unpause
      m_Paused = false;
    }
  }

}



void Game::PanOnPlayer()
{

  if ( m_pPlayer == NULL )
  {
    return;
  }
  if ( ( m_Level.Width() == Dobbs::LEVEL_DEFAULT_WIDTH )
  &&   ( m_Level.Height() == Dobbs::LEVEL_DEFAULT_HEIGHT ) )
  {
    m_Level.m_OffsetX = 0;
    m_Level.m_OffsetY = 0;
  }
  else
  {
    m_Level.m_OffsetX = m_pPlayer->GetX() - 400;
    if ( m_Level.m_OffsetX < 0 )
    {
      m_Level.m_OffsetX = 0;
    }
    if ( m_Level.m_OffsetX > m_Level.Width() * Dobbs::TILE_WIDTH - 800 )
    {
      m_Level.m_OffsetX = m_Level.Width() * Dobbs::TILE_WIDTH - 800;
    }

    m_Level.m_OffsetY = m_pPlayer->GetY() - 332;
    if ( m_Level.m_OffsetY < 0 )
    {
      m_Level.m_OffsetY = 0;
    }
    if ( m_Level.m_OffsetY > m_Level.Height() * Dobbs::TILE_HEIGHT - 608 )
    {
      m_Level.m_OffsetY = m_Level.Height() * Dobbs::TILE_HEIGHT - 608;
    }
  }

}



bool Game::IsTileSign( Dobbs::TileType Tile )
{

  return ( ( Tile == Dobbs::TILE_SCREEN_SIGN )
  ||       ( Tile == Dobbs::TILE_SIGN ) );

}



void Game::Render()
{

  g_App.RenderTextureSection( "Background", 0, 0, 0xff808080 );

  DWORD   ModColor = 0xffffffff;

  if ( m_AlertActive )
  {
    int     ColorByte = 0xc0 + ( (int)( 63 * cosf( m_AlertPos * 3.1415926f / 180.0f ) ) );
    ModColor = 0xffff0000 + ( ColorByte << 8 ) + ColorByte;
  }

  m_Level.Render( ModColor );

  // show collected tokens
  m_pEntityTokens->Render( 0, 0 );
  char    Temp[200];
  wsprintfA( Temp, "%d", m_CollectedIcons );
  g_App.RenderText( "GUI.Small", Temp, 65, 29, 0xff000000 );
  g_App.RenderText( "GUI.Small", Temp, 64, 28 );

  if ( m_pPlayer )
  {
    if ( ( IsTileSign( m_Level.TileBehindEntity( m_pPlayer ) ) )
    ||   ( IsTileSign( m_Level.TileBehindEntity( m_pPlayer, 0, -1 ) ) ) )
    {
      g_App.RenderTextureSection( "GUI.InGame", 144, 100, 0xa0ffffff );

      RECT    Area;

      SetRect( &Area, 280, 120, 280 + 370, 120 + 214 );
      g_App.RenderTextInArea( "GUI.Small", GetSignText(), Area, 0xff000000 );
      OffsetRect( &Area, -1, -1 );
      g_App.RenderTextInArea( "GUI.Small", GetSignText(), Area );
    }
  }
  if ( m_GetReadyDelay > 0.0f )
  {
    g_App.RenderTextureSection( "Game.GetReady", 398 / 2, 69 / 2 );
  }
  /*
  else if ( m_Victory )
  {
    g_App.RenderTextureSection( "Game.Victory", 335 / 2, 68 / 2 );
  }
  else if ( m_isDead )
  {
    g_App.RenderTextureSection( "Game.Dead", 373 / 2, 68 / 2 );
  }
  */

  if ( m_Paused )
  {
    g_App.RenderTextureSection( "Dimmer", 0, 0, 0x80000000 );
    g_App.RenderTextCentered( "GUI.Small", "Return to the Main Menu?  [Y/N]", 400, 230 );
  }

}



void Game::ReadMessage( int LevelPos, const std::string& SoundFile )
{

  if ( ( m_pSoundMessage != NULL )
  &&   ( m_pSoundMessage->IsPlaying() ) )
  {
    // don't start a new message while an old one is playing
    return;
  }
  if ( m_ReadMessages[LevelPos].find( LevelPos ) != m_ReadMessages[LevelPos].end() )
  {
    return;
  }
  if ( m_PreReadMessages[LevelPos].find( LevelPos ) != m_PreReadMessages[LevelPos].end() )
  {
    return;
  }

  if ( m_pSoundMessage != NULL )
  {
    delete m_pSoundMessage;
    m_pSoundMessage = NULL;
  }

  m_PreReadMessages[LevelPos].insert( LevelPos );
  m_pSoundMessage = new Sound();
  m_pSoundMessage->PreloadSound( SoundFile.c_str() );
  m_pSoundMessage->Play();

}



std::string Game::GetSignText()
{

  int   LevelPos = ( m_pPlayer->GetX() / 32 ) + ( m_pPlayer->GetY() / 32 ) * m_Level.Width();
  if ( IsTileSign( m_Level.TileBehindEntity( m_pPlayer, 0, -1 ) ) )
  {
    LevelPos -= m_Level.Width();
  }
  switch ( m_LevelNr )
  {
    case 1:
      if ( LevelPos == 911 )
      {
        ReadMessage( LevelPos, "investigate.wav" );
        return "Welcome Dr. Dobb.\r\nThere is a small disturbance in area B-1. Please investigate.";
      }
      else if ( LevelPos == 934 )
      {
        ReadMessage( LevelPos, "upjump.wav" );
        return "Press UP to jump";
      }
      else if ( LevelPos == 953 )
      {
        ReadMessage( LevelPos, "collecticons.wav" );
        return "Collect the Visual Studio icons";
      }
      else if ( LevelPos == 974 )
      {
        ReadMessage( LevelPos, "iconterminal.wav" );
        return "The terminal shows the number of icons needed. Press SPACE in front of it to active it";
      }
      else if ( LevelPos == 988 )
      {
        ReadMessage( LevelPos, "enterdoor.wav" );
        return "Well done! Press SPACE to enter a doorway";
      }
      break;
    case 2:
      if ( LevelPos == 3577 )
      {
        ReadMessage( LevelPos, "proceedtop.wav" );
        return "This is the gateway to the IO routines. Proceed to the top. Beware of free running bugs.";
      }
      else if ( LevelPos == 2947 )
      {
        ReadMessage( LevelPos, "donottouchbugs.wav" );
        return "Do not touch bugs. You can jump over them or throw an icon at them by pressing SPACE.";
      }
      else if ( LevelPos == 568 )
      {
        ReadMessage( LevelPos, "ioroutines.wav" );
        return "Proceed through this doorway to the IO routines.";
      }
      break;
    case 3:
      if ( LevelPos == 791 )
      {
        ReadMessage( LevelPos, "distahead.wav" );
        return "The disturbance is up ahead. Debug it carefully.";
      }
      break;
    case 4:
      if ( LevelPos == 3742 )
      {
        ReadMessage( LevelPos, "ioroutinesb1.wav" );
        return "These are the IO routines. Proceed to exit B 1. The antivirus software is alerted. Be careful.";
      }
      else if ( LevelPos == 1659 )
      {
        ReadMessage( LevelPos, "proceeddoorway.wav" );
        return "Process through this doorway to area B-1.";
      }
      else if ( LevelPos == 4670 )
      {
        ReadMessage( LevelPos, "doorlocked.wav" );
        return "The alarm locked down this doorway. You have to find another way.";
      }
      else if ( LevelPos == 4620 )
      {
        ReadMessage( LevelPos, "backdoor.wav" );
        return "You are lucky. Use this switch to access a backdoor to the upper routines.";
      }
      else if ( LevelPos == 2059 )
      {
        ReadMessage( LevelPos, "goodwork.wav" );
        return "Good work. You managed to survive. Proceed to area B1 through this gateway.";
      }
      break;
    case 5:
      if ( LevelPos == 1609 )
      {
        ReadMessage( LevelPos, "usecar.wav" );
        return "Use the car to go along the upper area. Press down to get out of the car again.";
      }
      break;
    case 6:
      if ( LevelPos == 3591 )
      {
        ReadMessage( LevelPos, "hurrydelete.wav" );
        return "The bugs have infected the antivirus software. You have to hurry.";
      }
      break;
    case 7:
      if ( LevelPos == 3015 )
      {
        ReadMessage( LevelPos, "booster.wav" );
        return "The yellow object ahead is a USB booster. Jump into it and press SPACE to launch again.";
      }
      break;
    case 8:
      if ( LevelPos == 1988 )
      {
        ReadMessage( LevelPos, "deletium.wav" );
        return "There is a pool of deletium ahead. Do not touch it without proper protection.";
      }
      else if ( LevelPos == 2043 )
      {
        ReadMessage( LevelPos, "subaccess.wav" );
        return "The access to the submarine is denied. Proceed to the next area and raise your security level to 2.";
      }
      break;
    case 10:
      if ( LevelPos == 9914 )
      {
        ReadMessage( LevelPos, "enterdiary.wav" );
        return "You have found my diary room. Enter and find the truth written.";
      }
      break;
    case 11:
      if ( LevelPos == 390 )
      {
        ReadMessage( LevelPos, "comefar.wav" );
        return "You shouldn't have come that far. This was not expected.";
      }
      else if ( LevelPos == 2481 )
      {
        ReadMessage( LevelPos, "accessdenied.wav" );
        return "Access denied.";
      }
      break;
    case 17:
      if ( LevelPos == 370 )
      {
        ReadMessage( LevelPos, "diary1.wav" );
        return "Well played, player. You have earned a reward. Go on and learn.";
      }
      else if ( LevelPos == 870 )
      {
        ReadMessage( LevelPos, "diary2.wav" );
        return "A long time ago things were easy. Weak systems with easy access.";
      }
      else if ( LevelPos == 1577 )
      {
        ReadMessage( LevelPos, "diary3.wav" );
        return "Bugs soared and allowed nearly unlimited freedom.";
      }
      else if ( LevelPos == 1608 )
      {
        ReadMessage( LevelPos, "diary4.wav" );
        return "But technology advanced and better solutions appeared that helped debugging.";
      }
      else if ( LevelPos == 406 )
      {
        ReadMessage( LevelPos, "diary5.wav" );
        return "Sometimes i wish i was back in my C64.";
      }
      break;
  }

  char    Temp[200];

  wsprintfA( Temp, "Position %d", LevelPos );
  return Temp;

}



void Game::OnCollision( Entity* pEntity1, Entity* pEntity2 )
{

  switch ( pEntity1->GetType() )
  {
    case Dobbs::ENTITY_TYPE_THROWN_TOKEN:
      if ( ( pEntity2->GetType() == Dobbs::ENTITY_TYPE_BUG_L )
      ||   ( pEntity2->GetType() == Dobbs::ENTITY_TYPE_BUG_R )
      ||   ( pEntity2->GetType() == Dobbs::ENTITY_TYPE_FLY )
      ||   ( pEntity2->GetType() == Dobbs::ENTITY_TYPE_SHODAN )
      ||   ( pEntity2->GetType() == Dobbs::ENTITY_TYPE_FISH_L )
      ||   ( pEntity2->GetType() == Dobbs::ENTITY_TYPE_FISH_R ) )
      {
        EntityThrownToken* pToken = (EntityThrownToken*)pEntity1;
        if ( !pToken->HasLanded() )
        {
          // kill enemy
          if ( pEntity2->Hit() )
          {
            pEntity2->Hurt();
            pEntity1->Die();
            if ( pEntity2->CanBeRemoved() )
            {
              PreCollect( pEntity2->GetSpawnX(), pEntity2->GetSpawnY() );
            }

            g_App.PlaySound( "Player.Explode" );
          }
        }
      }
      else if ( pEntity2->GetType() == Dobbs::ENTITY_TYPE_ANOMALY )
      {
        EntityThrownToken* pToken = (EntityThrownToken*)pEntity1;
        if ( !pToken->HasLanded() )
        {
          // kill enemy
          pEntity2->Hit();
          pEntity2->Hurt();
          pEntity1->Die();
          if ( pEntity2->CanBeRemoved() )
          {
            PreCollect( pEntity2->GetSpawnX(), pEntity2->GetSpawnY() );
          }

          m_Level.SpawnEntity( Dobbs::ENTITY_TYPE_EXPLOSION, 0, pEntity1->GetX(), pEntity1->GetY() );
          g_App.PlaySound( "Player.Explode" );
        }
      }
      break;
    case Dobbs::ENTITY_TYPE_SUB_SHOT:
      if ( ( pEntity2->GetType() == Dobbs::ENTITY_TYPE_FISH_L )
      ||   ( pEntity2->GetType() == Dobbs::ENTITY_TYPE_FISH_R )
      ||   ( pEntity2->GetType() == Dobbs::ENTITY_TYPE_FLY )
      ||   ( pEntity2->GetType() == Dobbs::ENTITY_TYPE_SHODAN )
      ||   ( pEntity2->GetType() == Dobbs::ENTITY_TYPE_LASER_D )
      ||   ( pEntity2->GetType() == Dobbs::ENTITY_TYPE_LASER_U )
      ||   ( pEntity2->GetType() == Dobbs::ENTITY_TYPE_LASER_L )
      ||   ( pEntity2->GetType() == Dobbs::ENTITY_TYPE_LASER_R ) )
      {
        // kill enemy
        pEntity2->Hit();
        pEntity2->Hurt();
        if ( pEntity2->CanBeRemoved() )
        {
          m_PreCollectedItems[m_LevelNr].insert( std::make_pair( pEntity2->GetSpawnX() / 32, pEntity2->GetSpawnY() / 32 ) );
        }
        pEntity1->Die();
      }
      else if ( pEntity2->GetType() == Dobbs::ENTITY_TYPE_ANOMALY )
      {
        // kill enemy
        pEntity2->Hit();
        pEntity2->Hurt();
        pEntity1->Die();
        if ( pEntity2->CanBeRemoved() )
        {
          PreCollect( pEntity2->GetSpawnX(), pEntity2->GetSpawnY() );
        }

        m_Level.SpawnEntity( Dobbs::ENTITY_TYPE_EXPLOSION, 0, pEntity1->GetX(), pEntity1->GetY() );
        g_App.PlaySound( "Player.Explode" );
      }
      break;
    case Dobbs::ENTITY_TYPE_SUB:
      if ( ( m_pPlayer == NULL )
      ||   ( m_pPlayer->GetState() != Dobbs::STATE_IN_SUB ) )
      {
        break;
      }
      // fall through to player (sub acts as player if Dr.Dobbs is inside)
    case Dobbs::ENTITY_TYPE_PLAYER:
      if ( ( pEntity2->GetType() == Dobbs::ENTITY_TYPE_TOKEN )
      ||   ( pEntity2->GetType() == Dobbs::ENTITY_TYPE_THROWN_TOKEN )
      ||   ( pEntity2->GetType() == Dobbs::ENTITY_TYPE_SECURITY_PASS )
      ||   ( pEntity2->GetType() == Dobbs::ENTITY_TYPE_KEY_BLUE )
      ||   ( pEntity2->GetType() == Dobbs::ENTITY_TYPE_KEY_GREEN )
      ||   ( pEntity2->GetType() == Dobbs::ENTITY_TYPE_KEY_RED )
      ||   ( pEntity2->GetType() == Dobbs::ENTITY_TYPE_KEY_YELLOW ) )
      {
        if ( pEntity2->CanBePicked() )
        {
          // Collect token
          m_PreCollectedItems[m_LevelNr].insert( std::make_pair( pEntity2->GetX() / 32, pEntity2->GetY() / 32 ) );
          pEntity2->Die();
          g_App.PlaySound( "Collect" );
          if ( ( pEntity2->GetType() == Dobbs::ENTITY_TYPE_TOKEN )
          ||   ( pEntity2->GetType() == Dobbs::ENTITY_TYPE_THROWN_TOKEN ) )
          {
            ++m_CollectedIcons;
          }
          else if ( pEntity2->GetType() == Dobbs::ENTITY_TYPE_SECURITY_PASS )
          {
            // open yellow doors
            Dobbs::TileType   OldDoor = Dobbs::TILE_DOOR_YELLOW;
            switch ( pEntity2->m_ExtraData )
            {
              case 1:
                OldDoor = Dobbs::TILE_DOOR_RED;
                break;
              case 2:
                OldDoor = Dobbs::TILE_DOOR_BLUE;
                break;
              case 3:
                OldDoor = Dobbs::TILE_DOOR_GREEN;
                break;
            }
            m_Level.ReplaceTile( OldDoor, Dobbs::TILE_DOOR_OPENED );
            m_PreItemCollected[pEntity2->m_ExtraData] = true;
          }
        }
      }
      else if ( pEntity2->GetType() == Dobbs::ENTITY_TYPE_CAR )
      {
        // enter car
        pEntity2->Carry( pEntity1 );
      }
      else if ( pEntity2->GetType() == Dobbs::ENTITY_TYPE_SUB )
      {
        // enter sub
        m_pPlayer->IntoSub( pEntity2 );
      }
      else if ( pEntity2->GetType() == Dobbs::ENTITY_TYPE_CANNON )
      {
        // enter cannon
        m_pPlayer->IntoCannon( pEntity2 );
      }
      else if ( ( pEntity2->GetType() == Dobbs::ENTITY_TYPE_BUG_L )
      ||        ( pEntity2->GetType() == Dobbs::ENTITY_TYPE_BUG_R )
      ||        ( pEntity2->GetType() == Dobbs::ENTITY_TYPE_FISH_L )
      ||        ( pEntity2->GetType() == Dobbs::ENTITY_TYPE_FISH_R )
      ||        ( pEntity2->GetType() == Dobbs::ENTITY_TYPE_SHODAN )
      ||        ( pEntity2->GetType() == Dobbs::ENTITY_TYPE_SHOT )
      ||        ( pEntity2->GetType() == Dobbs::ENTITY_TYPE_LASER_H )
      ||        ( pEntity2->GetType() == Dobbs::ENTITY_TYPE_LASER_V ) )
      {
        if ( !m_GameWon )
        {
          m_pPlayer->Die();
          m_pPlayer = NULL;
        }
      }
      break;
  }

}



void Game::Warp( int Level, int X, int Y )
{

  if ( Level != m_LevelNr )
  {
    m_LevelNr = Level;
    m_Level.m_Entities.remove( m_pPlayer );

    m_AlertActive = false;

    // warping acts as checkpoint
    m_CollectedItems    = m_PreCollectedItems;
    m_ReadMessages      = m_PreReadMessages;
    m_PlayerStartX      = X;
    m_PlayerStartY      = Y;
    m_PlayerStartTokens = m_CollectedIcons;

    for ( int i = 0; i < 4; ++i )
    {
      m_ItemCollected[i] = m_PreItemCollected[i];
    }

    // TODO - save progress


    LoadLevel();

    m_pPlayer->SetPosition( X, Y );
    m_Level.InsertEntity( m_pPlayer );
    m_Level.m_pPlayer = m_pPlayer;
    PanOnPlayer();
    m_Level.m_pPlayer->WarpIn();
  }

}



void Game::WalkThroughDoor( Entity* pDoor )
{

  int   Pos = ( pDoor->GetX() / 32 ) + ( pDoor->GetY() / 32 ) * m_Level.Width();

  switch ( m_LevelNr )
  {
    case 1:
      if ( Pos == 650 )
      {
        Warp( 3, 216, 479 );
        return;
      }
      else if ( Pos == 991 )
      {
        Warp( 2, 80, 3167 );
        return;
      }
      break;
    case 2:
      if ( Pos == 572 )
      {
        Warp( 4, 1268, 1246 );
        return;
      }
      else if ( Pos == 591 )
      {
        // secret door
        Warp( 15, 207, 1407 );
        return;
      }
      else if ( Pos == 3881 )
      {
        Warp( 1, 2922, 351 );
        return;
      }
      break;
    case 3:
      if ( Pos == 330 )
      {
        Warp( 1, 1661, 255 );
        return;
      }
      else if ( Pos == 3413 )
      {
        // to the car
        Warp( 5, 180, 319 );
        return;
      }
      else if ( Pos == 3375 )
      {
        // to the car
        Warp( 5, 180, 319 );
        return;
      }
      else if ( Pos == 725 )
      {
        // back to 4
        Warp( 4, 1953, 701 );
        return;
      }
      else if ( Pos == 2655 )
      {
        // back to 4
        Warp( 4, 1953, 701 );
        return;
      }
      break;
    case 4:
      if ( Pos == 1962 )
      {
        Warp( 3, 217, 478 );
        return;
      }
      else if ( Pos == 3738 )
      {
        // back to 2
        Warp( 2, 385, 509 );
        return;
      }
      break;
    case 5:
      if ( Pos == 1192 )
      {
        Warp( 6, 326, 4636 );
        return;
      }
      else if ( Pos == 1404 )
      {
        Warp( 3, 508, 1853 );
        return;
      }
      else if ( Pos == 6003 )
      {
        // TODO - back to 3
        Warp( 3, 508, 1853 );
        return;
      }
      break;
    case 6:
      if ( Pos == 190 )
      {
        Warp( 7, 384, 1023 );
        return;
      }
      else if ( Pos == 1272 )
      {
        // secret door
        Warp( 16, 190, 221 );
        return;
      }
      else if ( Pos == 3586 )
      {
        // back to 5
        Warp( 5, 6235, 799 );
        return;
      }
      break;
    case 7:
      if ( Pos == 4694 )
      {
        // secret door
        Warp( 15, 1713, 221 );
        return;
      }
      else if ( Pos == 903 )
      {
        // normal exit
        Warp( 8, 167, 413 );
        return;
      }
      else if ( Pos == 3009 )
      {
        // back to 6
        Warp( 6, 470, 319 );
        return;
      }
      break;
    case 8:
      if ( Pos == 3413 )
      {
        // security level 2 area
        Warp( 9, 1305, 2462 );
        return;
      }
      else if ( Pos == 10973 )
      {
        // to level 10
        Warp( 10, 272, 989 );
        return;
      }
      else if ( Pos == 1982 )
      {
        // back to level 7
        Warp( 7, 179, 383 );
        return;
      }
      break;
    case 9:
      if ( Pos == 5444 )
      {
        // exit to level 8
        Warp( 8, 5507, 638 );
        return;
      }
      break;
    case 10:
      if ( Pos == 1033 )
      {
        // exit to level 11
        Warp( 11, 903, 254 );
        return;
      }
      else if ( Pos == 4355 )
      {
        // back to level 8
        Warp( 8, 5522, 1983 );
        return;
      }
      else if ( Pos == 9910 )
      {
        // secret diary room
        Warp( 17, 194, 255 );
        return;
      }
      break;
    case 11:
      if ( Pos == 2474 )
      {
        // exit to level 12
        Warp( 12, 218, 1435 );
        return;
      }
      break;
    case 15:
      if ( Pos == 354 )
      {
        // exit to level 7
        Warp( 7, 2982, 1533 );
        return;
      }
      else if ( Pos == 2524 )
      {
        // exit to level 2
        Warp( 2, 993, 509 );
        return;
      }
      break;
    case 16:
      if ( Pos == 503 )
      {
        // exit to level 6
        Warp( 6, 730, 1661 );
        return;
      }
      else if ( Pos == 5594 )
      {
        // exit to level 6
        Warp( 6, 730, 1661 );
        return;
      }
      break;
    case 17:
      if ( Pos == 363 )
      {
        // exit to level 10
        Warp( 10, 400, 2175 );
        return;
      }
      break;
  }
  //dh::Log( "Warp through door %d/%d", m_LevelNr, Pos );

}



void Game::OnGameEvent( Dobbs::GameEvent GEvent )
{

  switch ( GEvent )
  {
    case Dobbs::GE_PLAYER_DIED:
      m_pPlayer = NULL;
      m_Level.m_pPlayer = NULL;
      break;
    case Dobbs::GE_SHODAN_KILLED:
      m_GameWon = true;
      m_WinDelay = 10.0f;
      break;
    case Dobbs::GE_STOP_ALARM_ANOMALY:
      // TODO - check for other anomalies
      {
        std::list<Entity*>::iterator    it( m_Level.m_Entities.begin() );
        while ( it != m_Level.m_Entities.end() )
        {
          Entity*   pEntity( *it );

          if ( ( !pEntity->CanBeRemoved() )
          &&   ( pEntity->GetType() == Dobbs::ENTITY_TYPE_ANOMALY ) )
          {
            // there is still an anomaly alive
            return;
          }

          ++it;
        }
      }
      // fall through
    case Dobbs::GE_STOP_ALARM:
      // turn off alarm
      if ( m_AlertActive )
      {
        m_AlertActive = false;
        // Alert - toggle lockdown doors
        for ( int i = 0; i < m_Level.Width(); ++i )
        {
          for ( int j = 0; j < m_Level.Height(); ++j )
          {
            Dobbs::Tile&    FullTile( m_Level.FullTile( i, j ) );

            if ( FullTile.Type == Dobbs::TILE_LOCKDOWN )
            {
              FullTile.Type = Dobbs::TILE_LOCKDOWN_OPEN;
            }
            else if ( FullTile.Type == Dobbs::TILE_LOCKDOWN_OPEN )
            {
              FullTile.Type = Dobbs::TILE_LOCKDOWN;
            }
          }
        }
        std::list<Entity*>::iterator    itE( m_Level.m_Entities.begin() );
        while ( itE != m_Level.m_Entities.end() )
        {
          Entity*   pEntity( *itE );

          if ( ( pEntity->GetType() == Dobbs::ENTITY_TYPE_LASER_D )
          ||   ( pEntity->GetType() == Dobbs::ENTITY_TYPE_LASER_U )
          ||   ( pEntity->GetType() == Dobbs::ENTITY_TYPE_LASER_R )
          ||   ( pEntity->GetType() == Dobbs::ENTITY_TYPE_LASER_L ) )
          {
            EntityLaser*    pLaser = (EntityLaser*)pEntity;
            pLaser->m_Active = false;
          }

          ++itE;
        }
      }
      break;
    case Dobbs::GE_START_ALARM:
      // start alarm
      if ( !m_AlertActive )
      {
        m_AlertActive = true;
        g_App.PlaySound( "Alert" );
        // Alert - toggle lockdown doors
        for ( int i = 0; i < m_Level.Width(); ++i )
        {
          for ( int j = 0; j < m_Level.Height(); ++j )
          {
            Dobbs::Tile&    FullTile( m_Level.FullTile( i, j ) );

            if ( FullTile.Type == Dobbs::TILE_LOCKDOWN )
            {
              FullTile.Type = Dobbs::TILE_LOCKDOWN_OPEN;
            }
            else if ( FullTile.Type == Dobbs::TILE_LOCKDOWN_OPEN )
            {
              FullTile.Type = Dobbs::TILE_LOCKDOWN;
            }
          }
        }
        std::list<Entity*>::iterator    itE( m_Level.m_Entities.begin() );
        while ( itE != m_Level.m_Entities.end() )
        {
          Entity*   pEntity( *itE );

          if ( ( pEntity->GetType() == Dobbs::ENTITY_TYPE_LASER_D )
          ||   ( pEntity->GetType() == Dobbs::ENTITY_TYPE_LASER_U )
          ||   ( pEntity->GetType() == Dobbs::ENTITY_TYPE_LASER_R )
          ||   ( pEntity->GetType() == Dobbs::ENTITY_TYPE_LASER_L ) )
          {
            EntityLaser*    pLaser = (EntityLaser*)pEntity;
            pLaser->m_Active = true;
          }
          if ( pEntity->GetType() == Dobbs::ENTITY_TYPE_SHODAN )
          {
            EntityShodan*    pShodan = (EntityShodan*)pEntity;
            pShodan->m_Active = true;
          }

          ++itE;
        }
      }
      break;
    case Dobbs::GE_ENABLE_LAYER_1:
      m_Level.EnableLayer( 0 );
      break;
    case Dobbs::GE_ENABLE_LAYER_2:
      m_Level.EnableLayer( 1 );
      break;
    case Dobbs::GE_ENABLE_LAYER_3:
      m_Level.EnableLayer( 2 );
      break;
    case Dobbs::GE_DISABLE_LAYER_1:
      m_Level.EnableLayer( 0, false );
      break;
    case Dobbs::GE_DISABLE_LAYER_2:
      m_Level.EnableLayer( 1, false );
      break;
    case Dobbs::GE_DISABLE_LAYER_3:
      m_Level.EnableLayer( 2, false );
      break;
    case Dobbs::GE_TOGGLE_LAYER_1:
      m_Level.EnableLayer( 0, !m_Level.LayerEnabled( 0 ) );
      break;
    case Dobbs::GE_TOGGLE_LAYER_2:
      m_Level.EnableLayer( 1, !m_Level.LayerEnabled( 1 ) );
      break;
    case Dobbs::GE_TOGGLE_LAYER_3:
      m_Level.EnableLayer( 2, !m_Level.LayerEnabled( 2 ) );
      break;
  }

}



void Game::Restart()
{

  m_PreCollectedItems = m_CollectedItems;
  m_PreReadMessages   = m_ReadMessages;
  for ( int i = 0; i < 4; ++i )
  {
    m_PreItemCollected[i] = m_ItemCollected[i];
  }

  m_AlertActive       = false;

  if ( m_pPlayer != NULL )
  {
    if ( m_pPlayer->Carrier() )
    {
      m_pPlayer->Carrier()->DropOff( m_pPlayer );
    }
    m_Level.m_Entities.remove( m_pPlayer );
  }

  LoadLevel();

  if ( m_pPlayer == NULL )
  {
    m_pPlayer = (EntityPlayer*)m_Level.SpawnEntity( Dobbs::ENTITY_TYPE_PLAYER, 0, m_PlayerStartX, m_PlayerStartY );
  }
  else
  {
    m_pPlayer->SetPlayerState( Dobbs::STATE_OK );
    m_Level.InsertEntity( m_pPlayer );
    m_pPlayer->SetPosition( m_PlayerStartX, m_PlayerStartY );
  }
  m_Level.m_pPlayer = m_pPlayer;
  PanOnPlayer();

  m_CollectedIcons = m_PlayerStartTokens;

  m_Level.m_pPlayer->WarpIn();

}



void Game::OnAppPaused( bool Pause )
{

  if ( Pause )
  {
    m_Paused = true;
  }

}



void Game::StoreSubPosition( int X, int Y )
{

  m_StoredSubPositions[m_LevelNr] = std::make_pair( X, Y );

}



void Game::PreCollect( int X, int Y )
{

  m_PreCollectedItems[m_LevelNr].insert( std::make_pair( X / 32, Y / 32 ) );

}